'''
Generate igor-shaded world tiles for the (most of) the world.
Assumes you have the SRTM tiles of area of interest.

Loads in SRTM data for the world in slices and generates hillshaded web
tiles for the data from zoom 7 to 11. Higher zooms are possible, but each 
lower zoom level requires loading 4x more SRTM data into RAM. 

Another solution will need to be worked out to generate lower zooms.

Each slice overlaps the other by 1 degree in order to prevent gaps in the tile data.

Run with a slice number as a command line parameter and will generate a script starting at that file.
Useful to continue after a crash.

'''
import os
import argparse
import time
from datetime import timedelta

#defaults which can be modified via command-line params
maperitive_path = "G:\\Maperitive\\"
script_file = "gen_terrain.mscript"
tiles_dir   = "G:\\Tiles"

#These are the full bounds of the SRTMGL1 data
map_bounds = {
    'W':-180,
    'E':180,
    'S':-56,
    'N':59,
}

script_head = '''
// SCRIPT TO GENERATE WORLD TERRAIN SHADING TILES
// Expects SRTM1 data in Maperitive\Cache\Rasters\SRTM1
//   Which should be a symlink to \SRTMGL1_full
// Will output tiles to {tdir}

set-dem-source name=SRTM1
'''

script_foot = '''
'''

script_gen = '''
//Tile {tnum} of {ttot}
clear-map
set-geo-bounds {Wb},{Sb},{Eb},{Nb}
generate-relief-igor color="black" intensity=3 sample-rate=1
//generate-contours interval=10
set-geo-bounds {W},{S},{E},{N}
generate-tiles exclude-partial=true minzoom=7 maxzoom=11 tilesdir={tdir}
'''

def gen_bounds(start=1):
    #generate the maperitive script metadata
    print "Generating script starting at slice", start

    #Break the map into chunks using 6 degree steps.
    #With 6 degree steps, 2 degree overlap and 1 degree bleed is required to generate zoom 7.
    EW_step = 6 #longitude
    NS_step = 6 #latitude
    overlap = 2 #How much to overlap each slice of the world
    bleed   = 1 #How much extra data to load (Should be 1)

    EW_range = range(map_bounds['W'], map_bounds['E']-(EW_step-overlap)+1, (EW_step-overlap)) 
    NS_range = range(map_bounds['S'], map_bounds['N']-(NS_step-overlap)+1, (NS_step-overlap))

    mem = EW_step * NS_step * (3601*3601*2)
    mem_bleed = (2*bleed + EW_step) * (2*bleed + NS_step) * (3601*3601*2)
    total = len(EW_range) * len(NS_range)
    print "Dividing world into {} slices ({:,} bytes each, max {:,} with bleed)".format(total,mem,mem_bleed)

    metadata = []
    num = 1
    for EW in EW_range:
        for NS in NS_range:
            bounds = {
                'tnum': num,
                'ttot': total,
                'W':max(map_bounds['W']+.00001, EW),
                'E':min(map_bounds['E']-.00001, EW + EW_step),
                'S':max(map_bounds['S']+.00001, NS),
                'N':min(map_bounds['N']-.00001, NS + NS_step),                
                'Wb':max(map_bounds['W'], EW-bleed),
                'Eb':min(map_bounds['E'], EW+EW_step+bleed),
                'Sb':max(map_bounds['S'], NS-bleed),
                'Nb':min(map_bounds['N'], NS+NS_step+bleed),
            }

            if num == start:
                print "Start tile: {W},{S},{E},{N}".format(**bounds)
            if num >= start:
                metadata.append(bounds)

            num += 1
    return metadata

def gen_script(data, args):
    # Generate the maperitive script
    print "Writing script..."
    print os.path.join(args.mdir, "Scripts", args.script)
    with open(os.path.join(args.mdir, "Scripts", args.script), "w") as F:
        F.write(script_head.format(tdir=args.tdir))
        for elem in data:
            F.write(script_gen.format(tdir=args.tdir, **elem))
        F.write(script_foot)

def run_script(data, args):
    #Generate one block of the script at a time and run it
    #If there's an error (such as out of memory) start again
    print "Running script..."
    t_tot = 0;
    t_last = 0;
    for elem in data:
        t_start = time.time()
        with open(os.path.join(args.mdir, "Scripts", args.script), "w") as F:
            F.write(script_head.format(tdir=args.tdir))
            F.write(script_gen.format(tdir=args.tdir, **elem))
        
        path = os.path.join(args.mdir,"Maperitive.Console.exe")
        script = os.path.join("Scripts", args.script)

        done = False
        while not done:
            print "*"*80
            print ("Slice %d of %d"%(elem['tnum'], elem['ttot'])).center(80, "*")
            t_str = "Time Elapsed: %s"%timedelta(seconds=t_tot)
            if t_last > 0:
                t_str += "-- Last Slice: %s"%timedelta(seconds=t_last)
            print t_str.center(80,"*")
            print "*"*80

            err = os.system(path + " " + script)
            print "Error:", err
            if err == 0:
                done = True
        t_end = time.time()
        t_last = t_end - t_start
        t_tot += t_last


def main(args):
    data = gen_bounds(args.index)
    if args.run:
        run_script(data, args)
    else:
        gen_script(data, args)

if __name__ == '__main__':
    P = argparse.ArgumentParser(description="Generates a maperitive script to create a terrain map of the world")
    P.add_argument("-i", "--index", type=int, default=1,
         help="Index to start at")
    P.add_argument("-r", "--run", action="store_true",
        help="Add this argument to run maperitive also")
    P.add_argument("--mdir", default=maperitive_path, 
        help="Path to the Maperitive folder to write the script to. Default: %s"%maperitive_path)
    P.add_argument("--tdir", default=tiles_dir, 
        help="Path to output the generated tiles. Default: %s"%tiles_dir)
    P.add_argument("--script", default=script_file, 
        help="File name of the script to generate. Default: %s"%script_file)
    args = P.parse_args()

    main(args)